<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\Investors;
use App\Models\Deal;
use App\Models\User;
use App\Models\DealInvestments;
use App\Models\InvestorTax;

class InvestorsController extends Controller {

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct() {
        $this->middleware('auth');
        $this->middleware('role:admin'); // Ensure only admin users can access this controller
    }

    public function add() {
        return view('admin.investor.add');
    }

    public function saveInvestor(Request $request) {
        // Validate the request data
        $request->validate([
            'firstname' => 'required|string',
            'lastname' => 'required|string',
            'phone' => 'required|numeric|min:13',
            'email' => 'required|email|unique:users',
            'password' => 'required|string|confirmed',
        ]);

        // Create a user record in the users table
        $user = User::create([
                'name' => $request->input('firstname') . ' ' . $request->input('lastname'),
                'email' => $request->input('email'),
                'password' => Hash::make($request->input('password')),
        ]);
        
        $investorid = str_pad(mt_rand(1, 999999), 6, '0', STR_PAD_LEFT);

        // Create an investor record in the investors table if needed
        Investors::create([
            'user_id' => $user->id,
            'investorid' => $investorid,
            'firstname' => $request->input('firstname'),
            'lastname' => $request->input('lastname'),
            'email' => $user->email,
            'phone' => $request->input('phone'),
        ]);
        
        return redirect()->route('list.investors')->with('success', 'Investor added successfully');
    }

    /**
     * Show the dashboard of investors list.
     *
     */
    public function list() {

        $investors = Investors::with('user')->orderBy('created_at', 'desc')->paginate(10); // 10 records per page, adjust as needed
        return view('admin.investor.list', compact('investors'));
    }

    public function addInvestment() {

        $investors = User::where('roleid', 2)->where('is_active', 1)->get();
        $deals = Deal::where('status', 1)->get();
        return view('admin.investor.add_investment', compact('investors', 'deals'));
    }

    public function assignDealToInvestor(Request $request) {
        // Validate the request data
        $validatedData = $request->validate([
            'deal_id' => 'required|integer',
            'investor_id.*' => 'required|integer',
            'invested_amt.*' => 'required|numeric',
            'invested_on.*' => 'required|date',
        ]);

        // Get the authenticated user's ID
        $updatedBy = Auth::id();

        // Loop through the submitted data to create or update investments
        foreach ($validatedData['investor_id'] as $key => $investorId) {
            $investment = DealInvestments::updateOrCreate(
                            [
                                'deal_id' => $validatedData['deal_id'],
                                'investor_id' => $investorId,
                            ],
                            [
                                'invested_amt' => $validatedData['invested_amt'][$key],
                                'invested_on' => $validatedData['invested_on'][$key],
                                'updated_by' => $updatedBy, // Assign the logged-in user's ID
                            ]
            );
        }

        // Redirect or return a response as needed
        return redirect()->route('admin.deal.deatails', $validatedData['deal_id'])->with('success', 'Investment details updated successfully');
    }

    /**
     * Remove the specified deal from the database.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyInvestorDeal($id) {
        $deal = DealInvestments::findOrFail($id);
        $deal->delete();
        return redirect()->back()->with('success', 'Investor deleted successfully');
    }

    public function getInvestmentData(Request $request) {

        $dealInvestments = DealInvestments::with('user') // Eager load the associated user
                ->where('id', $request->investment_id) // Filter by deal_id
                ->get();

        // Return the dealInvestments data as JSON response
        return response()->json($dealInvestments);
    }

    public function updateInvestmentData(Request $request) {
        // Validate the incoming request data (you can add validation rules as needed)
        $validatedData = $request->validate([
            'invested_amt' => 'required|numeric',
            'invested_on' => 'required|date',
        ]);

        // Get the investment record to update (you may need to adjust this based on how you identify the record)
        $investment = DealInvestments::find($request->investment_id);

        if (!$investment) {
            return response()->json(['message' => 'Investment not found'], 404);
        }

        // Update the investment record with the new data
        $investment->update([
            'invested_amt' => $validatedData['invested_amt'],
            'invested_on' => $validatedData['invested_on'],
        ]);

        // Return a success response (you can customize this as needed)
        return response()->json(['message' => 'Investment updated successfully']);
    }

    public function addInvestorTaxes() {

        $investors = User::where('roleid', 2)->get();
        $deals = Deal::where('status', 1)->get();
        return view('admin.investor.tax_documents', compact('investors', 'deals'));
    }
    public function storDocuments(Request $request) {
        // Validate the form data
        $request->validate([
            'investor_id' => 'required',
            'deal_id' => 'required|exists:deals,id', // Ensure the deal_id exists in the deals table
            'file_name.*' => 'required|string|max:255', // Use ".*" to validate each element in the array
            'tax_file.*' => 'required|file|mimes:pdf', // Use ".*" to validate each element in the array
        ]);
        
        //dd($request->all());

        // Process and store each uploaded file
        if ($request->hasFile('tax_file')) {
            $uploadedFiles = $request->file('tax_file');
            $documentNames = $request->input('file_name');

            foreach ($uploadedFiles as $key => $file) {
                // Generate a unique filename
                $filename = time() . '_' . uniqid('document_') . '.' . $file->getClientOriginalExtension();

                // Define the destination directory
                $destinationPath = public_path('uploads/deals/taxes');

                // Use the move method to save the file to the destination directory
                $file->move($destinationPath, $filename);

                // Create a new DealDocument record for each file with its respective document name
                InvestorTax::create([
                    'investor_id' => $request->input('investor_id'),
                    'deal_id' => $request->input('deal_id'),
                    'file_name' => $documentNames[$key], // Use the document name from the array
                    'tax_file' => $filename,
                    'status' => 1, // You can set the default status here
                ]);
            }
        }

        // Redirect back with a success message
        return redirect()->route('admin.deals.list')->with('success', 'Documents added successfully');
    }
    
}
